function [t_array dt] = makeTime(kgrid, c, cfl)
%MAKETIME   Create an evenly spaced array of time points.
%
% DESCRIPTION:
%       makeTime creates an evenly spaced array of time points for use with
%       the first-order k-space simulation codes based on the
%       Courant-Friedrichs-Lewy stability level cfl and the grid size. The
%       time step dt is chosen based on cfl (the default setting is 0.3),
%       and the number of time steps is chosen based on the time it takes
%       to travel from one corner of the k-space grid given by kgrid to the
%       geometrically opposite corner travelling at c m/s. 
%
% USAGE:
%       [t_array dt] = makeTime(kgrid, c)
%       [t_array dt] = makeTime(kgrid, c, cfl)
%
% INPUTS:
%       kgrid       - kspace grid structure returned by makeGrid
%       c           - maximum sound speed in the medium [m/s]
%
% OPTIONAL INPUTS:
%       cfl         - Courant-Friedrichs-Lewy stability criterion 
%                     (default = 0.3) 
%
% OUTPUTS:
%       t_array     - array of evenly-spaced time points [s]
%       dt          - time step [s]
%
% ABOUT:
%       author      - Bradley Treeby
%       date        - 3rd July 2009
%       last update - 5th October 2009
%       
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010 Bradley Treeby and Ben Cox
%
% See also makeGrid, kspaceFirstOrder1D, kspaceFirstOrder2D,
% kspaceFirstOrder3D 

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>.

% CHANGE LOG:
% 2009/10/05    - changed pixel_dim to select min pixel size instead of max

% assign optional inputs
if nargin < 3
    % define the Courant-Friedrichs-Lewy stability level
    cfl = 0.3;
end

% if c is a matrix, take the maximum value
c = max(c(:));

% set the simulation length based on the size of the grid diagonal and
% the maximum sound speed in the medium
switch numDim(kgrid.k)
    case 1
        pixel_dim = kgrid.dx;
        t_end = kgrid.x_size/c;
    case 2
        pixel_dim = min([kgrid.dz, kgrid.dx]);
        t_end = sqrt(kgrid.z_size.^2 + kgrid.x_size.^2)/c;
    case 3
        pixel_dim = min([kgrid.dz, kgrid.dx, kgrid.dy]);
        t_end = sqrt(kgrid.z_size.^2 + kgrid.x_size.^2 + kgrid.y_size.^2)/c;
end
       
% assign a time step based on Courant stability criterion
dt = cfl*pixel_dim/c;

% create the time array
t_array = 0:dt:t_end;